<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
// PENTING: Library Midtrans
use Midtrans\Config;
use Midtrans\Snap;

class OrderController extends Controller
{
    // 1. Tampilkan Riwayat Pesanan
    public function index()
    {
        $orders = Order::where('user_id', Auth::id())
                        ->with('items.product') 
                        ->latest()
                        ->get();

        return view('orders.index', compact('orders'));
    }

    // 2. Proses Checkout dengan Midtrans
    public function checkout()
    {
        $user = Auth::user();
        $carts = Cart::where('user_id', $user->id)->get();

        if ($carts->isEmpty()) {
            return redirect()->back()->with('error', 'Keranjang kosong.');
        }

        // --- KONFIGURASI MIDTRANS ---
        // CONTOH: 'SB-Mid-server-aBcD12345...'
		Config::$serverKey = 'Mid-server-WbsD6JLMCp0Rqr7DaeWNwWix'; 
		Config::$isProduction = false;
		Config::$isSanitized = true;
		Config::$is3ds = true;

        // Buat Nomor Order Unik
        $orderNumber = 'ORD-' . time() . '-' . rand(100,999);

        // --- DATABASE TRANSACTION ---
        $order = DB::transaction(function () use ($user, $carts, $orderNumber) {
            
            // A. Hitung Total
            $totalPrice = $carts->sum(function($item) {
                return $item->product->price * $item->qty;
            });

            // B. Buat Order (Status: pending)
            $newOrder = Order::create([
                'user_id' => $user->id,
                'total_price' => $totalPrice,
                'status' => 'pending', 
            ]);

            // C. Pindahkan Item
            foreach ($carts as $cart) {
                OrderItem::create([
                    'order_id' => $newOrder->id,
                    'product_id' => $cart->product_id,
                    'qty' => $cart->qty,
                    'price' => $cart->product->price,
                ]);

                // Kurangi Stok
                $product = Product::find($cart->product_id);
                if($product) {
                    $product->decrement('stock', $cart->qty);
                }
            }

            // D. Hapus Keranjang
            Cart::where('user_id', $user->id)->delete();
            
            return $newOrder;
        });

        // --- MINTA TOKEN KE MIDTRANS ---
        // --- MINTA TOKEN KE MIDTRANS ---
        $params = [
            'transaction_details' => [
                'order_id' => $order->id, // Pastikan ID ini Unik!
                'gross_amount' => (int) $order->total_price,
            ],
            'customer_details' => [
                'first_name' => $user->name,
                'email' => $user->email,
            ],
            // TAMBAHAN: Agar QRIS/E-Wallet muncul spesifik
            'enabled_payments' => [
                'gopay',        // Ini akan men-generate QRIS GoPay
                'qris',         // QRIS Umum (ShopeePay, Dana, LinkAja support ini)
                'shopeepay',
                'bank_transfer', // Opsional: jika mau tetap ada transfer bank
                'credit_card'
            ],
        ];

        try {
            $snapToken = Snap::getSnapToken($params);
            $order->update(['snap_token' => $snapToken]);
            
            return redirect()->route('orders.index')->with('success', 'Pesanan dibuat. Silakan bayar sekarang!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal memproses pembayaran: ' . $e->getMessage());
        }
    }

    // 3. Callback Sukses Bayar (Sederhana)
    public function paymentSuccess($id)
    {
        $order = Order::findOrFail($id);
        $order->update(['status' => 'paid']); // Ubah jadi Lunas

        return redirect()->route('orders.index')->with('success', 'Pembayaran Berhasil! Terima kasih.');
    }

    // --- BAGIAN ADMIN ---

    public function indexAdmin()
    {
        if (Auth::user()->role !== 'admin') { abort(403); }
        $orders = Order::with('user')->latest()->get();
        return view('admin.orders.index', compact('orders'));
    }

    public function updateStatus(Request $request, $id)
    {
        if (Auth::user()->role !== 'admin') { abort(403); }
        $order = Order::findOrFail($id);
        
        $request->validate([
            'status' => 'required|in:pending,paid,shipped,completed,cancelled'
        ]);

        $order->update(['status' => $request->status]);

        return redirect()->back()->with('success', 'Status pesanan diperbarui!');
    }
}