<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use Illuminate\Support\Facades\File; // Wajib import ini untuk hapus file manual

class ProductController extends Controller
{
    // 1. Tampilkan Daftar Produk
    public function index()
    {
        $products = Product::latest()->get();
        return view('admin.products.index', compact('products'));
    }

    // 2. Tampilkan Form Tambah
    public function create()
    {
        return view('admin.products.create');
    }

    // 3. Simpan Produk Baru (STORE)
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'price' => 'required|numeric',
            'stock' => 'required|numeric',
            'description' => 'nullable',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $input = $request->all();

        // --- LOGIKA UPLOAD KE FOLDER 'gambar' ---
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            // Buat nama unik (contoh: 17098822.jpg)
            $filename = time() . '.' . $file->getClientOriginalExtension();
            
            // Pindahkan file ke folder public/gambar
            $file->move(public_path('gambar'), $filename);
            
            // Simpan nama filenya saja ke database
            $input['image'] = $filename;
        }

        Product::create($input);

        return redirect()->route('admin.products')->with('success', 'Produk berhasil ditambahkan!');
    }

    // 4. Tampilkan Form Edit
    public function edit($id)
    {
        $product = Product::findOrFail($id);
        return view('admin.products.edit', compact('product'));
    }

    // 5. Update Produk (UPDATE) - INI YANG PENTING AGAR GAMBAR TIDAK HILANG
    public function update(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $request->validate([
            'name' => 'required',
            'price' => 'required|numeric',
            'stock' => 'required|numeric',
            'description' => 'nullable',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // Boleh kosong saat edit
        ]);

        $input = $request->all();

        // --- LOGIKA CEK GAMBAR SAAT UPDATE ---
        if ($request->hasFile('image')) {
            // A. Hapus gambar lama jika ada
            $gambarLama = public_path('gambar/' . $product->image);
            if ($product->image && File::exists($gambarLama)) {
                File::delete($gambarLama);
            }

            // B. Upload gambar baru
            $file = $request->file('image');
            $filename = time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('gambar'), $filename);
            
            $input['image'] = $filename;
        } else {
            // C. JIKA TIDAK UPLOAD GAMBAR BARU
            // Hapus key 'image' dari input agar database tidak di-update dengan NULL
            unset($input['image']);
        }

        $product->update($input);

        return redirect()->route('admin.products')->with('success', 'Produk berhasil diperbarui!');
    }

    // 6. Hapus Produk
    public function destroy($id)
    {
        $product = Product::findOrFail($id);

        // Hapus file gambar dari folder public/gambar
        $gambarLama = public_path('gambar/' . $product->image);
        if ($product->image && File::exists($gambarLama)) {
            File::delete($gambarLama);
        }

        $product->delete();

        return redirect()->route('admin.products')->with('success', 'Produk berhasil dihapus!');
    }
}